C      *****************************************************************
C      * Subroutine FindIndex                                          *
C      * Finds the indices for a set of species-of-interest in the     *
C      * species concentration array SpConc. (These indices are the    *
C      * same as the indices in the Labels array.)  These species-of-  *
C      * interest are those that the tree model is concerned with.     *
C      * The soil will contain other species in addition to those of   *
C      * interest to the tree model.                                   *
C      * An array of the names of the species-of-interest has been set *
C      * up.  The labels array is then checked for each of these       *
C      * species in turn.  When a match is found the corresponding     *
C      * index is saved.  When all species-of-interest have been found *
C      * their labels' indices are stored in a species index array.    *
C      * If a species-of-interest is not found in the labels array, its*
C      * index will be -1.                                             *
C      * Note that due to the lack of functions which can manipulate   *
C      * character variables, the names of the species of interest and *
C      * the labels must match exactly (i.e. uppercase and lowercase). *
C      * Regular chemical notation is used, e.g. calcium is written Ca,*
C      * not CA.                                                       *
C      *                                                               *
C      * Variables:                                                    *
C      * Done      -Local - Indicates whether the species of interest  *
C      *                    has been found in the labels array so that *
C      *                    needless checking through the labels array *
C      *                    is avoided. (Done=0 means not done, Done=1 *
C      *                    means done.)  Unitless.                    *
C      * I         -Local - Array index.  Unitless.                    *
C      * J         -Local - Array index.  Unitless.                    *
C      * Labels    -Input - A vector, of size NSpecies, of the name for*
C      *                    each species.  Unitless.                   *
C      *                    (Common block VModel, file VModel.f)       *
C      * NSpecies  -Input - The number of species in the system (i.e.  *
C      *                    the number of components plus the number   *
C      *                    of dependent species).  Unitless.          *
C      *                    (Common block VModel, file VModel.f)       *
C      * SpIndex   -Output- A vector, of size SpNum, of indices into   *
C      *                    arrays which match each species of         *
C      *                    interest.  SpIndex(i) is an index that when*
C      *                    used in the Labels array will give the     *
C      *                    name of the species amount the tree has    *
C      *                    consumed is NuTake(i).  SpIndex(i) when    *
C      *                    used as the index into SpConc will give the*
C      *                    concentration of this species.  Input from *
C      *                    TREGRO, or read into the model by the      *
C      *                    routine InParam.  Unitless.                *
C      *                    (Common block VIF, file VModel.f)          *
C      * SpName    -Input - A vector, of size SpNum, of names of       *
C      *                    species of interest.  Input from TREGRO, or*
C      *                    read into the model by the routine InParam.*
C      *                    Unitless.                                  *
C      *                    (Common block VIF, file VModel.f)          *
C      * SpNum     -Input - The number of species of interest.  Input  *
C      *                    from TREGRO, or read into the model by the *
C      *                    routine InParam.  Unitless.                *
C      *                    (Common block VIF, file VModel.f)          *
C      *****************************************************************
       SUBROUTINE FINDINDEX
							IMPLICIT NONE
							INCLUDE 'VMODEL.INC'
							INTEGER DONE, I, J

C      *---------------------------------------------------------------*
C      * Initialize all indices to an impossible index so that it will *
C      * be obvious if a species-of-interest is not in the soil        *
C      * chemistry model.                                              *
C      *---------------------------------------------------------------*
       DO 100 I=1, SPNUM
	         SPINDEX(I) = -1
  100  CONTINUE		  

C      *-------------------------------------------------------------*
C      * Find each species of interest in the list of species in the *
C      * soil chemistry model and save the corresponding index.      *
C      * Note that there will be more species in the soil chemistry  *
C      * model than in the species-of-interest list.                 *
C      * For sulfur and phosphorus only look at the first letter of  *
C      * soil chemistry name.  Note that the code do this for all the*
C      * species of interest but I have chosen to not do this, b/c   *
C      * there is already a case where there can be two types of     *
C      * nitrogen which are handled in another way (see NitroCheck). *
C      * This can always be changed in the future to act the same for*
C      * all species of interest.                                    *
C      *-------------------------------------------------------------*
       DO 110 I=1, SPNUM
          J = 1
	      	  DONE = 0
  120     CONTINUE		  
C            *---------------------------------------------------*
C            * If the species is sulfur or phosphorus then just  *
C            * look for the first letter of the soil model's     *
C            * element.                                          *
C            *---------------------------------------------------*
             IF ((SPNAME(I).EQ.'S').OR.(SPNAME(I).EQ.'P')) THEN
                IF (LABELS(J)(1:1).EQ.SPNAME(I)) THEN
         			       SPINDEX(I) = J
            	   			DONE = 1
                ELSE
                   J = J + 1
                ENDIF
C            *----------------------------------------------------*
C            * If the species is other than sulfur or phosphorus  *
C            * then look for the entire name of the soil model's  *
C            * element.                                           *
C            *----------------------------------------------------*
      		     ELSE 
                IF (LABELS(J).EQ.SPNAME(I)) THEN
         			       SPINDEX(I) = J
               				DONE = 1
        	   		  ELSE
        	   		     J = J + 1
	           		  ENDIF
             ENDIF
	      	  IF((J.LE.NSPECIES).AND.(DONE.EQ.0)) GOTO 120
  110  CONTINUE
  							
       RETURN
							END
C      *****************************************************************
C      *                    END SUBROUTINE                             *
C      *****************************************************************
